<?php
/**
 * Plugin Name: Grayscale Theme Switcher
 * Plugin URI: https://www.facebook.com/myphoenix168
 * Description: เปลี่ยนหน้าเว็บทั้งหมดให้เป็นโทนขาวดำและธีมสีต่างๆ พร้อมตัวเลือกการปรับแต่งขั้นสูง
 * Version: 1.0.0
 * Author: Jakkrit Sangakong
 * Author URI: https://www.facebook.com/myphoenix168
 * License: GPL v2 or later
 * Text Domain: grayscale-theme
 */

// ป้องกันการเข้าถึงโดยตรง
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('GTS_VERSION', '1.0.0');
define('GTS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('GTS_PLUGIN_URL', plugin_dir_url(__FILE__));

class GrayscaleThemeSwitcher {
    
    private $options;
    
    public function __construct() {
        $this->options = get_option('gts_settings');
        
        // Hook actions
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('wp_footer', array($this, 'add_theme_switcher_button'));
        
        // AJAX handlers
        add_action('wp_ajax_gts_save_preference', array($this, 'save_user_preference'));
        add_action('wp_ajax_nopriv_gts_save_preference', array($this, 'save_user_preference'));
    }
    
    public function init() {
        // Load text domain for translations
        load_plugin_textdomain('grayscale-theme', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function enqueue_frontend_assets() {
        // เช็คว่า plugin ถูกเปิดใช้งานหรือไม่
        if (!isset($this->options['enabled']) || !$this->options['enabled']) {
            return;
        }
        
        // Enqueue CSS
        wp_enqueue_style(
            'gts-styles',
            GTS_PLUGIN_URL . 'assets/gts-styles.css',
            array(),
            GTS_VERSION
        );
        
        // Enqueue JavaScript
        wp_enqueue_script(
            'gts-script',
            GTS_PLUGIN_URL . 'assets/gts-script.js',
            array('jquery'),
            GTS_VERSION,
            true
        );
        
        // Get user saved preference
        $user_theme = isset($_COOKIE['gts_user_theme']) ? sanitize_text_field($_COOKIE['gts_user_theme']) : null;
        $user_intensity = isset($_COOKIE['gts_user_intensity']) ? intval($_COOKIE['gts_user_intensity']) : null;
        
        // Check force override
        $force_override = isset($this->options['force_override']) ? $this->options['force_override'] : false;
        
        // Localize script
        wp_localize_script('gts-script', 'gts_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('gts_nonce'),
            'default_theme' => isset($this->options['default_theme']) ? $this->options['default_theme'] : 'normal',
            'allow_user_toggle' => isset($this->options['allow_user_toggle']) ? $this->options['allow_user_toggle'] : true,
            'intensity' => isset($this->options['intensity']) ? $this->options['intensity'] : 100,
            'user_theme' => $user_theme,
            'user_intensity' => $user_intensity,
            'force_override' => $force_override,
            'force_override_timestamp' => isset($this->options['force_override_timestamp']) ? $this->options['force_override_timestamp'] : 0,
            'preferences_cleared' => isset($this->options['preferences_cleared']) ? $this->options['preferences_cleared'] : 0,
            'debug' => defined('WP_DEBUG') && WP_DEBUG
        ));
        
        // Add inline CSS for dynamic settings
        $this->add_dynamic_css();
    }
    
    private function add_dynamic_css() {
        $intensity = isset($this->options['intensity']) ? $this->options['intensity'] : 100;
        $default_theme = isset($this->options['default_theme']) ? $this->options['default_theme'] : 'normal';
        
        $custom_css = '';
        
        if ($default_theme === 'grayscale') {
            $custom_css .= "
                html.gts-grayscale {
                    filter: grayscale({$intensity}%) !important;
                }
            ";
        } elseif ($default_theme === 'sepia') {
            $custom_css .= "
                html.gts-sepia {
                    filter: sepia({$intensity}%) !important;
                }
            ";
        } elseif ($default_theme === 'dark') {
            $custom_css .= "
                html.gts-dark {
                    filter: invert({$intensity}%) hue-rotate(180deg) !important;
                }
                html.gts-dark img,
                html.gts-dark video,
                html.gts-dark iframe {
                    filter: invert(100%) hue-rotate(180deg) !important;
                }
            ";
        } elseif ($default_theme === 'high-contrast') {
            $contrast_value = 100 + ($intensity / 2);
            $custom_css .= "
                html.gts-high-contrast {
                    filter: contrast({$contrast_value}%) !important;
                }
            ";
        }
        
        wp_add_inline_style('gts-styles', $custom_css);
    }
    
    public function add_theme_switcher_button() {
        if (!isset($this->options['enabled']) || !$this->options['enabled']) {
            return;
        }
        
        if (!isset($this->options['allow_user_toggle']) || !$this->options['allow_user_toggle']) {
            return;
        }
        
        $button_position = isset($this->options['button_position']) ? $this->options['button_position'] : 'bottom-right';
        $force_override = isset($this->options['force_override']) ? $this->options['force_override'] : false;
        ?>
        <div id="gts-theme-switcher" class="gts-position-<?php echo esc_attr($button_position); ?>">
            <button id="gts-toggle-btn" aria-label="เปลี่ยนธีมสี">
                <span class="gts-icon">🎨</span>
            </button>
            <div id="gts-theme-menu" class="gts-hidden">
                <h3>เลือกธีมสี</h3>
                <?php if ($force_override) : ?>
                <div style="background: #fff3cd; color: #856404; padding: 10px; margin-bottom: 10px; border-radius: 5px; font-size: 12px;">
                    ⚠️ ธีมถูกกำหนดโดยผู้ดูแลระบบ
                </div>
                <?php endif; ?>
                <div class="gts-theme-options">
                    <button class="gts-theme-option" data-theme="normal" <?php echo $force_override ? 'disabled' : ''; ?>>
                        <span class="gts-preview normal"></span>
                        <span>ปกติ</span>
                    </button>
                    <button class="gts-theme-option" data-theme="grayscale" <?php echo $force_override ? 'disabled' : ''; ?>>
                        <span class="gts-preview grayscale"></span>
                        <span>ขาวดำ</span>
                    </button>
                    <button class="gts-theme-option" data-theme="sepia" <?php echo $force_override ? 'disabled' : ''; ?>>
                        <span class="gts-preview sepia"></span>
                        <span>ซีเปีย</span>
                    </button>
                    <button class="gts-theme-option" data-theme="dark" <?php echo $force_override ? 'disabled' : ''; ?>>
                        <span class="gts-preview dark"></span>
                        <span>โหมดมืด</span>
                    </button>
                    <button class="gts-theme-option" data-theme="high-contrast" <?php echo $force_override ? 'disabled' : ''; ?>>
                        <span class="gts-preview contrast"></span>
                        <span>คอนทราสต์สูง</span>
                    </button>
                </div>
                <?php if (isset($this->options['allow_intensity_control']) && $this->options['allow_intensity_control']) : ?>
                <div class="gts-intensity-control">
                    <label for="gts-intensity-slider">ความเข้ม: <span id="gts-intensity-value">100</span>%</label>
                    <input type="range" id="gts-intensity-slider" min="0" max="100" value="100" <?php echo $force_override ? 'disabled' : ''; ?>>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    public function save_user_preference() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'gts_nonce')) {
            wp_die('Security check failed');
        }
        
        $theme = sanitize_text_field($_POST['theme']);
        $intensity = intval($_POST['intensity']);
        
        // Save to cookie
        setcookie('gts_user_theme', $theme, time() + (86400 * 30), '/');
        setcookie('gts_user_intensity', $intensity, time() + (86400 * 30), '/');
        
        wp_send_json_success();
    }
    
    public function add_admin_menu() {
        add_options_page(
            'Grayscale Theme Switcher Settings',
            'Theme Switcher',
            'manage_options',
            'gts-settings',
            array($this, 'settings_page')
        );
    }
    
    public function settings_init() {
        register_setting('gts_settings_group', 'gts_settings', array($this, 'sanitize_settings'));
        
        add_settings_section(
            'gts_main_settings',
            'การตั้งค่าหลัก',
            array($this, 'settings_section_callback'),
            'gts-settings'
        );
        
        // Enable/Disable Plugin
        add_settings_field(
            'gts_enabled',
            'เปิดใช้งาน Plugin',
            array($this, 'enabled_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Default Theme
        add_settings_field(
            'gts_default_theme',
            'ธีมเริ่มต้น',
            array($this, 'default_theme_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Force Override User Preferences
        add_settings_field(
            'gts_force_override',
            'บังคับใช้ธีมเริ่มต้น',
            array($this, 'force_override_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Allow User Toggle
        add_settings_field(
            'gts_allow_user_toggle',
            'อนุญาตให้ผู้ใช้เปลี่ยนธีม',
            array($this, 'allow_user_toggle_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Button Position
        add_settings_field(
            'gts_button_position',
            'ตำแหน่งปุ่ม',
            array($this, 'button_position_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Default Intensity
        add_settings_field(
            'gts_intensity',
            'ความเข้มเริ่มต้น (%)',
            array($this, 'intensity_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Allow Intensity Control
        add_settings_field(
            'gts_allow_intensity_control',
            'แสดงตัวควบคุมความเข้ม',
            array($this, 'allow_intensity_control_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
        
        // Clear All User Preferences
        add_settings_field(
            'gts_clear_preferences',
            'ล้างการตั้งค่าผู้ใช้',
            array($this, 'clear_preferences_field_callback'),
            'gts-settings',
            'gts_main_settings'
        );
    }
    
    public function settings_section_callback() {
        echo '<p>กำหนดค่าการทำงานของ Grayscale Theme Switcher</p> Credit : <a href="https://www.facebook.com/myphoenix168">Jakkrit Sangakong</a>';
    }
    
    public function enabled_field_callback() {
        $enabled = isset($this->options['enabled']) ? $this->options['enabled'] : false;
        ?>
        <input type="checkbox" name="gts_settings[enabled]" value="1" <?php checked($enabled, 1); ?>>
        <label>เปิดใช้งานการเปลี่ยนธีม</label>
        <?php
    }
    
    public function default_theme_field_callback() {
        $default_theme = isset($this->options['default_theme']) ? $this->options['default_theme'] : 'normal';
        ?>
        <select name="gts_settings[default_theme]">
            <option value="normal" <?php selected($default_theme, 'normal'); ?>>ปกติ</option>
            <option value="grayscale" <?php selected($default_theme, 'grayscale'); ?>>ขาวดำ</option>
            <option value="sepia" <?php selected($default_theme, 'sepia'); ?>>ซีเปีย</option>
            <option value="dark" <?php selected($default_theme, 'dark'); ?>>โหมดมืด</option>
            <option value="high-contrast" <?php selected($default_theme, 'high-contrast'); ?>>คอนทราสต์สูง</option>
        </select>
        <?php
    }
    
    public function allow_user_toggle_field_callback() {
        $allow = isset($this->options['allow_user_toggle']) ? $this->options['allow_user_toggle'] : true;
        ?>
        <input type="checkbox" name="gts_settings[allow_user_toggle]" value="1" <?php checked($allow, 1); ?>>
        <label>แสดงปุ่มให้ผู้ใช้เปลี่ยนธีมได้เอง</label>
        <?php
    }
    
    public function button_position_field_callback() {
        $position = isset($this->options['button_position']) ? $this->options['button_position'] : 'bottom-right';
        ?>
        <select name="gts_settings[button_position]">
            <option value="bottom-right" <?php selected($position, 'bottom-right'); ?>>ล่างขวา</option>
            <option value="bottom-left" <?php selected($position, 'bottom-left'); ?>>ล่างซ้าย</option>
            <option value="top-right" <?php selected($position, 'top-right'); ?>>บนขวา</option>
            <option value="top-left" <?php selected($position, 'top-left'); ?>>บนซ้าย</option>
        </select>
        <?php
    }
    
    public function intensity_field_callback() {
        $intensity = isset($this->options['intensity']) ? $this->options['intensity'] : 100;
        ?>
        <input type="number" name="gts_settings[intensity]" value="<?php echo esc_attr($intensity); ?>" min="0" max="100">
        <p class="description">ค่าความเข้มของเอฟเฟกต์ (0-100)</p>
        <?php
    }
    
    public function allow_intensity_control_field_callback() {
        $allow = isset($this->options['allow_intensity_control']) ? $this->options['allow_intensity_control'] : true;
        ?>
        <input type="checkbox" name="gts_settings[allow_intensity_control]" value="1" <?php checked($allow, 1); ?>>
        <label>อนุญาตให้ผู้ใช้ปรับความเข้มได้</label>
        <?php
    }
    
    public function force_override_field_callback() {
        $force = isset($this->options['force_override']) ? $this->options['force_override'] : false;
        ?>
        <input type="checkbox" name="gts_settings[force_override]" value="1" <?php checked($force, 1); ?>>
        <label>บังคับใช้ธีมเริ่มต้นกับทุกคน (ไม่สนใจการตั้งค่าของผู้ใช้)</label>
        <p class="description">⚠️ เมื่อเปิดใช้ จะไม่สนใจการเลือกธีมของผู้ใช้และบังคับใช้ธีมเริ่มต้นเสมอ</p>
        <?php
    }
    
    public function clear_preferences_field_callback() {
        ?>
        <button type="button" id="gts-clear-all-preferences" class="button button-secondary">ล้างการตั้งค่าผู้ใช้ทั้งหมด</button>
        <p class="description">ล้างการตั้งค่าธีมที่ผู้ใช้เลือกไว้ทั้งหมด เพื่อให้ใช้ค่าเริ่มต้นใหม่</p>
        <div id="gts-clear-result" style="margin-top: 10px; display: none;"></div>
        <script>
        jQuery(document).ready(function($) {
            $('#gts-clear-all-preferences').on('click', function() {
                if (confirm('คุณแน่ใจหรือไม่ที่จะล้างการตั้งค่าผู้ใช้ทั้งหมด?\n\nการกระทำนี้จะทำให้ผู้ใช้ทุกคนต้องเลือกธีมใหม่')) {
                    // Set a flag to clear preferences
                    $('<input>').attr({
                        type: 'hidden',
                        name: 'gts_clear_all_preferences',
                        value: '1'
                    }).appendTo('form');
                    
                    $('#gts-clear-result').html('<span style="color: green;">✅ จะล้างการตั้งค่าผู้ใช้ทั้งหมดเมื่อบันทึก</span>').show();
                }
            });
        });
        </script>
        <?php
    }
    
    public function sanitize_settings($input) {
        $sanitized = array();
        
        // Sanitize all inputs
        $sanitized['enabled'] = isset($input['enabled']) ? 1 : 0;
        $sanitized['default_theme'] = sanitize_text_field($input['default_theme']);
        $sanitized['force_override'] = isset($input['force_override']) ? 1 : 0;
        $sanitized['allow_user_toggle'] = isset($input['allow_user_toggle']) ? 1 : 0;
        $sanitized['button_position'] = sanitize_text_field($input['button_position']);
        $sanitized['intensity'] = intval($input['intensity']);
        $sanitized['allow_intensity_control'] = isset($input['allow_intensity_control']) ? 1 : 0;
        
        // Check if we need to clear all user preferences
        if (isset($_POST['gts_clear_all_preferences']) && $_POST['gts_clear_all_preferences'] == '1') {
            // Add a timestamp to force clear on frontend
            $sanitized['preferences_cleared'] = time();
            add_settings_error(
                'gts_settings',
                'preferences_cleared',
                'การตั้งค่าผู้ใช้ทั้งหมดถูกล้างแล้ว',
                'success'
            );
        }
        
        // If force override is enabled, also set a timestamp to trigger clear
        if ($sanitized['force_override']) {
            $old_force = isset($this->options['force_override']) ? $this->options['force_override'] : false;
            if (!$old_force) {
                // Just enabled force override
                $sanitized['force_override_timestamp'] = time();
            } else {
                // Preserve existing timestamp
                $sanitized['force_override_timestamp'] = isset($this->options['force_override_timestamp']) ? $this->options['force_override_timestamp'] : time();
            }
        }
        
        return $sanitized;
    }
    
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1>Grayscale Theme Switcher Settings</h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('gts_settings_group');
                do_settings_sections('gts-settings');
                submit_button('บันทึกการตั้งค่า');
                ?>
            </form>
            
            <div class="gts-preview-section">
                <h2>ตัวอย่างธีม</h2>
                <div style="display: flex; gap: 20px; margin-top: 20px;">
                    <div style="text-align: center;">
                        <div style="width: 150px; height: 100px; background: linear-gradient(45deg, #ff6b6b, #4ecdc4); border: 1px solid #ddd;"></div>
                        <p>ปกติ</p>
                    </div>
                    <div style="text-align: center;">
                        <div style="width: 150px; height: 100px; background: linear-gradient(45deg, #ff6b6b, #4ecdc4); filter: grayscale(100%); border: 1px solid #ddd;"></div>
                        <p>ขาวดำ</p>
                    </div>
                    <div style="text-align: center;">
                        <div style="width: 150px; height: 100px; background: linear-gradient(45deg, #ff6b6b, #4ecdc4); filter: sepia(100%); border: 1px solid #ddd;"></div>
                        <p>ซีเปีย</p>
                    </div>
                    <div style="text-align: center;">
                        <div style="width: 150px; height: 100px; background: linear-gradient(45deg, #ff6b6b, #4ecdc4); filter: invert(100%) hue-rotate(180deg); border: 1px solid #ddd;"></div>
                        <p>โหมดมืด</p>
                    </div>
                    <div style="text-align: center;">
                        <div style="width: 150px; height: 100px; background: linear-gradient(45deg, #ff6b6b, #4ecdc4); filter: contrast(150%); border: 1px solid #ddd;"></div>
                        <p>คอนทราสต์สูง</p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}

// Initialize the plugin
function gts_init() {
    new GrayscaleThemeSwitcher();
}
add_action('plugins_loaded', 'gts_init');

// Activation hook
register_activation_hook(__FILE__, 'gts_activate');
function gts_activate() {
    // Set default options
    $default_options = array(
        'enabled' => true,
        'default_theme' => 'normal',
        'allow_user_toggle' => true,
        'button_position' => 'bottom-right',
        'intensity' => 100,
        'allow_intensity_control' => true
    );
    
    if (!get_option('gts_settings')) {
        add_option('gts_settings', $default_options);
    }
    
    // Create assets directory
    $upload_dir = wp_upload_dir();
    $plugin_dir = $upload_dir['basedir'] . '/gts-assets';
    if (!file_exists($plugin_dir)) {
        wp_mkdir_p($plugin_dir);
    }
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'gts_deactivate');
function gts_deactivate() {
    // Clean up if needed
}

// Uninstall hook
register_uninstall_hook(__FILE__, 'gts_uninstall');
function gts_uninstall() {
    // Remove options
    delete_option('gts_settings');
    
    // Remove any custom files if needed
    $upload_dir = wp_upload_dir();
    $plugin_dir = $upload_dir['basedir'] . '/gts-assets';
    if (file_exists($plugin_dir)) {
        // Remove directory and files
        array_map('unlink', glob("$plugin_dir/*"));
        rmdir($plugin_dir);
    }
}
